#pragma once

#include <memory>
#include <list>
#include <mutex>

namespace event::auto_update
{
	class actor;

	/* observer class */
	class observer
	{
		// actor can touch our private parts...
		friend class actor;

	public:

		// actor has been updated
		virtual void on_update(actor&) = 0;

		// dtor
		~observer(void);

	protected:

		// observe/unobserve given actor
		void observe(actor *p);
		void unobserve(actor* p);	

	private:
		std::recursive_mutex m_lock;
		std::list<actor*> m_list;
	};

	/* actor class */
	class actor
	{
		// observer can touch our private parts...
		friend class observer;

	public:

		// notify all observers on destruction
		~actor(void)
		{
			// copy list
			std::list<observer*> listcpy;

			{
				std::lock_guard<std::recursive_mutex> lk(this->m_lock);
				listcpy = this->m_list;
			}

			for (auto& p : listcpy)
				if (p != nullptr)
					p->unobserve(this);
		}

		// signal all observers we have been changed
		void signal(void)
		{
			std::lock_guard<std::recursive_mutex> lk(this->m_lock);

			for (auto& p : this->m_list)
				if (p != nullptr)
					p->on_update(*this);
		}

	private:

		// register a new observer
		void register_observer(observer *p)
		{
			std::lock_guard<std::recursive_mutex> lk(this->m_lock);

			this->m_list.push_back(p);
		}

		// unregister an observer
		void unregister_observer(observer *p)
		{
			std::lock_guard<std::recursive_mutex> lk(this->m_lock);

			this->m_list.remove(p);
		}

	private:
		std::recursive_mutex m_lock;
		std::list<observer*> m_list;
	};	
};