#pragma once

#include "../debug/safe.h"
#include "../storage/buffer.h"

#include "rle8.h"

namespace encoder
{
	namespace rle0
	{
		// RLE0 encoding
		static storage::buffer encode(const storage::buffer& buf)
		{
			// create temp buffer
			storage::buffer ret;

			// apply RLE8 first
			auto rle8 = encoder::rle8::encode(buf);

			// current block size
			size_t curr_block_size = 1;

			// scan RLE8 buffer
			size_t i = 0;

			while (i < rle8.size())
			{
				// unecoded region
				if (*rle8.ptr<unsigned char>(i) == 1)
				{
					// temp buffer for unencoded data
					storage::buffer unencoded;

					// scan next tokens
					size_t j = 0;

					while (MAKE_ADD(i, (j << 1)) < rle8.size() && j < 255)
					{
						auto tmp = MAKE_ADD(i, (j << 1));

						// skip if not unencoded anymore
						if (*rle8.ptr<unsigned char>(tmp) != 1)
							break;

						// add to unencoded buffer
						unencoded.append(rle8.ptr<unsigned char>(MAKE_ADD(tmp, (size_t)1)), 1, false);

						// increment j
						MAKE_INC(j, (size_t)1);
					}

					// number of bytes to write
					size_t write_size = unencoded.size();

					// evaluate if it worth replacing
					size_t original_bytes = unencoded.size() << 1;
					size_t new_bytes = MAKE_ADD(unencoded.size(), (size_t)3);

					// penalty if block size differs
					if (write_size != curr_block_size)
						MAKE_INC(new_bytes, (size_t)2);

					// replace if better
					if (new_bytes <= original_bytes)
					{
						// check if current block size differs from write size
						if (curr_block_size != write_size)
						{
							// push command
							ret.append<unsigned char>(0, false);
							ret.append<unsigned char>((unsigned char)write_size, false);

							curr_block_size = write_size;
						}

						// push content
						ret.append<unsigned char>(1, false);
						ret.append(unencoded.data(), write_size, false);
					}
					// copy identical otherwise
					else
					{
						// restore 1 bytes block size
						if (curr_block_size != 1)
						{
							ret.append<unsigned char>(0, false);
							ret.append<unsigned char>(1, false);

							curr_block_size = 1;
						}

						ret.append(rle8.ptr<unsigned char>(i), original_bytes, false);
					}

					// skip bytes
					MAKE_INC(i, original_bytes);
				}
				// otherelse copy data
				else
				{
					// restore 1 bytes block size
					if (curr_block_size != 1)
					{
						ret.append<unsigned char>(0, false);
						ret.append<unsigned char>(1, false);

						curr_block_size = 1;
					}

					// copy data
					ret.append(rle8.ptr<unsigned char>(i), 2, false);

					MAKE_INC(i, (size_t)2);
				}
			}

			// return new buffer
			return ret;
		}

		// RLE0 decode
		static storage::buffer decode(const storage::buffer& buf)
		{
			// create temp buffer
			storage::buffer ret;

			// current block size
			size_t curr_block_size = 1;

			// loop through file
			size_t i = 0;

			while (i < buf.size())
			{
				// get occurence
				auto occurence = *buf.ptr<unsigned char>(i++);

				// check if command
				if (occurence == 0)
				{
					// read new block size
					curr_block_size = (size_t)*buf.ptr<unsigned char>(i++);

					// read occurence
					occurence = *buf.ptr<unsigned char>(i++);
				}

				// get current block
				storage::buffer block(buf.ptr<unsigned char>(i), curr_block_size);

				// add N times
				while (occurence--)
					ret.append(block, false);

				// increment index
				i += curr_block_size;
			}

			// return buffer
			return ret;
		}
	};
};