#pragma once

#include <mutex>
#include <memory>

#include "exception.h"

#include "../utils/utils.h"
#include "../utils/singleton.h"
#include "../strings/format.h"
#include "../strings/fileparts.h"
#include "../threads/lock.h"

#ifdef _WIN32
#include <Windows.h>
#endif

 // use extended debugging
#define _EXDEBUG

#define EVEMON_CLASSNAME    "EVEMON"

// Event monitor types
enum class event_type
{
    DBG_MESSAGE = 0,
    DBG_WARNING = 1,
    DBG_ERROR = 2,
    DBG_CRITICAL = 3,
    DBG_EXCEPTION = 4,
    DBG_EXDEBUG = 5,
};

/* event_monitor singleton class */
class event_monitor : public singleton<event_monitor>
{
    friend class singleton<event_monitor>;

public:

    // notify message
    void notify(event_type type, const char* pszFilename, unsigned int uiLineNumber, const std::string& rMessage) const
    {
        std::lock_guard<std::mutex> lock(this->m_lock);

        auto filename = split_file_path(pszFilename).file;

        // always localy display in debug mode
#ifdef _DEBUG
        try
        {
            std::string line = _event2string(type) + format(" (%s:%d): ", filename.c_str(), uiLineNumber) + rMessage + std::string("\r\n");

#if defined(_WIN32) && !defined(_CONSOLE)
            OutputDebugStringA(line.c_str());
#endif

            switch (type)
            {
                case event_type::DBG_MESSAGE:
                    fprintf(stderr, "\033[32m");
                    break;

                case event_type::DBG_WARNING:
                    fprintf(stderr, "\033[33m");
                    break;

                case event_type::DBG_ERROR:
                    fprintf(stderr, "\033[31m");
                    break;

                case event_type::DBG_CRITICAL:
                    fprintf(stderr, "\033[1;31m");
                    break;

                case event_type::DBG_EXCEPTION:
                    fprintf(stderr, "\033[1;33m");
                    break;

                case event_type::DBG_EXDEBUG:
                    fprintf(stderr, "\033[3;32m");
                    break;
            }

            fprintf(stderr, "%s", line.c_str());
            fprintf(stderr, "\033[0m");
        }
        catch(...) {}
#endif
    }

private:

    // find window on initialization
    event_monitor(void) {}

    // convert Event type to a string
    static std::string _event2string(event_type type)
    {
        switch (type)
        {
            case event_type::DBG_MESSAGE:
                return "message";

            case event_type::DBG_WARNING:
                return "warning";

            case event_type::DBG_ERROR:
                return "error";

            case event_type::DBG_CRITICAL:
                return "critical";

            case event_type::DBG_EXCEPTION:
                return "exception";

            case event_type::DBG_EXDEBUG:
                return "extended debug";

            default:
                return "unknown";
        }
    }

    mutable std::mutex m_lock;
};

// macro to notify Events
#ifdef _MSC_VER

    #define _event(type, file, line, szfmt, ...)     ::get_instance<event_monitor>()->notify(type, file, line, format(szfmt, __VA_ARGS__))

    #define _debug(szfmt, ...)		                 _event(event_type::DBG_MESSAGE, __FILE__, __LINE__, szfmt, __VA_ARGS__);
    #define _warning(szfmt, ...)	                 _event(event_type::DBG_WARNING, __FILE__, __LINE__, szfmt, __VA_ARGS__);
    #define _error(szfmt, ...)		                 _event(event_type::DBG_ERROR, __FILE__, __LINE__, szfmt, __VA_ARGS__);
    #define _critical(szfmt, ...)	                 _event(event_type::DBG_CRITICAL, __FILE__, __LINE__, szfmt, __VA_ARGS__);

#else

    #define _event(type, file, line, szfmt, ...)	::get_instance<event_monitor>()->notify(type, file, line, format(szfmt __VA_OPT__(,) __VA_ARGS__))

    #define _debug(szfmt, ...)		                 _event(event_type::DBG_MESSAGE, __FILE__, __LINE__, szfmt __VA_OPT__(,) __VA_ARGS__);
    #define _warning(szfmt, ...)	                 _event(event_type::DBG_WARNING, __FILE__, __LINE__, szfmt __VA_OPT__(,) __VA_ARGS__);
    #define _error(szfmt, ...)		                 _event(event_type::DBG_ERROR, __FILE__, __LINE__, szfmt __VA_OPT__(,) __VA_ARGS__);
    #define _critical(szfmt, ...)	                 _event(event_type::DBG_CRITICAL, __FILE__, __LINE__, szfmt __VA_OPT__(,) __VA_ARGS__);

#endif